/*
 * dev_clinet_api.h
 *
 *  Created on: Jan 25, 2021
 *  Copyright by Xinhuan Intelligence co,.ltd 2021
 */

#ifndef SRC_DEV_API_CLIENT_API_H_
#define SRC_DEV_API_CLIENT_API_H_

#include "hi_comm_isp.h"

#ifdef __cplusplus
#if __cplusplus
extern "C" {
#endif
#endif /* End of #ifdef __cplusplus */


#define ALARM_OUT_ON  1
#define ALARM_OUT_OFF 0

#define SNAPSHOT_TIMEOUT 	119
#define OPEN_FILE_FAILUER 	120
#define SNAPSHOT_FILE_BAD 	121

#define SNAPSHOT_BUFFER_TYPE  0
#define SNAPSHOT_FILE_TYPE  1

#define MAX_CODES 12
#define MAX_PLATE_NO_BYTES 64

typedef struct PlateBox {
	float xmin;   // 左上角x轴坐标
	float ymin;   // 左上角y轴坐标
	float xmax;   // 右下角x轴坐标
	float ymax;   // 右下角y轴坐标
	float score;  // 车牌框置信度
} PlateBox;

typedef struct sSpeed {
	int value;
	int limitation;
	int status;
}Speed;

typedef struct sPlateBasicInfo {
	int label;			   				// 识别的目标类型，小于等于1是车牌，大于1其他目标
	float prob;            				// 车牌识别置信度
	PlateBox bbox;         				// 车牌位置 
	float confidences[MAX_CODES];  		// 车牌号码单字置信度数组
	int len;							// 车牌号码单字置信度数组长度
	char plateNos[MAX_PLATE_NO_BYTES];  // UTF8变长码编码的车牌号码。
	int plateNoBytes;     				// 车牌号码实际实际长度，不超过MAX_PLATE_NO_BYTES变量定义
	char plateColor[16];                //  白  黄  蓝  黑  绿  黄绿
	int plateColorBytes;				//  车牌颜色字节数
	int anchorBoxNo;					// 识别框编号，取值-1, 0, 1, 2,...，-1表示不支持此属性或者没有输出此属性

	/**
	 * 中国大陆车牌类型，定义如下
	 * 0 {未知}
	 * 10 - 16  {军 双层军牌 武警 双层武警牌 警 应急 新电动车}
	 * 20 - 23 {黄 教练车牌 双层黄牌 挂车车牌}
	 * 30 {蓝色小汽车}
	 * 40 - 43 {使馆牌 领馆牌 粤港澳牌 黑色小汽车 }
	 * 50 {大型双能源}
	 * 60 - 63  {新能源 民航  农用车牌 电动车牌（非机动）}
	 */
	int plateType ;	

	char deviceIP[15];

	int deviceIPbytes;

	/**
	 * 方向标志 -1 未定义 0 从上到下 1 从下到上
	 */
	int direction;

	/**
	 * 车辆事件状态；对于不同系列有不同的定义
	 * 对低位视频桩、新能源车位管理相机、高位视频桩产品定义如下：
	 * 0    空
	 * 1	进场
	 * 2	停稳
	 * 3	出场
	 *
	 * 21	抓拍 (通用)
	 *
	 * E5系列定义的状态如下：
	 * 22 入场
	 * 23 倒车
	 * 24 机动车滞留
	 * 25 非机动车滞留
	 * 26 行人滞留
	 * 27 无牌车
	 * 28 跟车
	 *
	 */
	int parkingSpotStatus =  -1;

	/**
	 * 车牌真伪标志 -1 未定义 0 假 1 真
	 * 
	 */
	int plateFlag = -1; 

	/**
	 * 车头车尾标志 -1 未定义 0 车头 1 车尾
	 */
	int headTailFlag = -1;
	 
	/**
	 *  E5系列产品有效
	 *  -1 未定义 0 相机视频触发 1 线圈触发 2 输入线路1 3 输入线路2 ...
	 *   
	 */
	int triggerMode = -1;

	int imageBufferSize;                // 图片buffer长度
	char * imageBuffer;				    // 图片Buffer, JPEG格式

} PlateBasicInfo;


typedef struct sDeviceInfo {
		char deviceName[64]; // 设备名称
		char ip[32];
		char firmware[16]; 	// 固件号
		char version[16];    // 版本
		char productNo[24] ; // 产品序列号
		int minVersion ;  // minor version
		unsigned long startup ;     // 设备启动时间戳
		char productType[16];       // 产品型号
} DeviceInfo;

typedef struct sBlackWhiteEntry {
	int type = 0 ;  					   // 0 黑名单 1 白名单
	char plateNos[MAX_PLATE_NO_BYTES];	   // UTF8变长码编码的车牌号码。
	int plateNoBytes = 0;				   // 车牌号码实际实际长度，不超过MAX_PLATE_NO_BYTES变量定义

	char plateColor[16];                   //  UTF8变长码 未知 白 黄 蓝 黑 黄绿 绿
	int plateColorBytes = 0;			   //  车牌颜色字节数

	int startTime = -1;						// 有效期开始的时间戳， -1表示永久有效
	int endTime = -1;						// 有效期截止的时间戳， -1表示永久有效

} BlackWhiteEntry ;

enum API_STATUS {
	OK,
	FAILED
};

typedef void (*API_CALLBACK)(PlateBasicInfo *);

typedef void (*CAMERA_EVENT_CALLBACK)(const char * ip, int code, const char * msg);

typedef void (*BWList_CALLBACK)(BlackWhiteEntry* entry);


/**
 ********* Basic working mechanism   ***********************************
 *
 * The license plate recognition device is referred to as
 * the device for short, and the device communicates with the client
 * through the TCP/IP protocol. In the process of client initializa-
 * tion, start the service of receiving license plate information.
 * After the device recognizes the license plate, it sends the lice-
 * nse plate recognition information to the client.
 *
 ***********************************************************************/

/*************************************************************************
 *                         基本工作机制
 * 这里,LLPR车牌识别相机简称为设备,设备和客户端通过TCP/IP进行通讯。客户端在初始化时，
 * 启动一个接收车牌信息的监听服务。当设备识别到车牌时，发送给监听服务。监听服务调用callback
 * 以实现信息的对接。
 * 
 *
 ************************************************************************/

/**
 * @brief initialize device communication API,
 * return OK if success else FAILED or a positive number.
 * 初始化客户端，初始化成功返回OK，否则返回FAILED或者一个正整数错误代码。
 *
 * @param local_ip[in] 运行在客户端的机器IP
 * @param local_ip[in] 运行在客户端的机器的端口，用于和相机进行通讯
 * @cb[in]  当客户端接收到车牌或者停车事件时，调用此接口
 */
int init_client(const char * local_ip, int local_port, API_CALLBACK cb);


/**
 * @brief
 * register a callback for receive camera online/offine event
 * callback's parameter code value is 0 represents the camera is online.
 *  other values represent camera has exception.
 * 注册相机上线下线事件回调
 * 回调函数的参数code值为0，表示相机正常运行，其他大于0的值，表示有异常。
 *
 */
void reg_camera_event_callback(CAMERA_EVENT_CALLBACK camera_vent_callback);


/**
 * @brief set a certification key for decrypt some confidential information
 * only support 16 bytes key
 */
void set_cert_key(char * buff, int len);


/**
 * @brief 连接某一个相机
 * 返回API_STATUS::OK表示连接成功，否则返回FAILED或者一个正整数错误代码。
 *
 * @param device_ip 相机IP地址
 */
int connect_device(const char * device_ip);

/**
 * @brief 断开所有的相机连接，关闭客户端
 *
 */
void close_client();

/***********************************************************************
 * below functions neither need initialize API nor close API
 ***********************************************************************/

/**
 * @brief reboot device, device will reboot in a few seconds
 * 发送重启指令给相机, 相机会在几秒之内重启
 *
 * @param device_ip 相机IP地址
 */
int reboot_device(const char * device_ip);

/**
 * @brief set camera's time
 * return OK if success else FAILED or a positive number.
 * 设置相机的时间
 * 返回API_STATUS::OK表示连接成功，否则返回FAILED或者一个正整数错误代码。
 * @param epoch unix标准的时间戳
 */
int set_time(const char * device_ip, int epoch);

/**
 * @brief trigger alarm out
 * parameter line value starts 0 represents the first alarm line.
 * the alarm is not be closed when milliseconds is less than 0. otherwise the alarm will close after milliseconds.
 *
 * 触发报警线路, line参数从0开始，代表第一路报警输出,
 * 如果milliseconds参数取值小于等0，则报警输出不会被关闭，
 * 如果milliseconds参数大于0，则在此值表示的毫秒之后，报警关闭
 * 调用成功返回OK，否则返回FAILED或者一个正整数错误代码。
 * @param device_ip 相机IP地址
 * @param line 报警线路
 * @milliseconds 报警线路持续的毫秒数
 *
 */
int trigger_alarmout(const char * device_ip, int line, int duration);

/**
 * @brief
 * query a current value of alarm line, then set the pointer of status if no exception
 * 查询相机的某一路报警线路的状态。
 * 如果此函数返回API_STATUS::OK，则表示查询成功。回写报警线路的值给status。
 *
 * @param device_ip 相机IP地址
 * @param line 报警线路
 * @param status[out] 的值为0，表示关闭状态， 1表示处于报警状态
 *
 */
int query_alarm(const char * device_ip, int line, int * status);

/**
 * @brief
 * write data to UART specified by UART alias, 485-1, 485-2, serial1, serial2, etc.
 * 向相机的UART端口写数据
 * @param device_ip 相机IP地址
 * @param alias: UART端口的别名，具体的值需要和相机的串口参数配置中的别名相同，比如485-1 485-2等 *
 * @param buff 待写入的数据缓冲
 * @param len  代写的数据长度
 *
 */
int write_uart(const char * device_ip, const char * alias, const char * buff, int len);

/**
 * @brief take a snapshot of a camera specified by device_ip;
 * the snapshot then the snapshot will be send to the client that connected
 * the camera. 
 * 向相机发送抓拍指令，相机接收到到抓拍指令后，返回成功，否则返回失败或者其他大于0的值
 * 注意：本函数的抓拍是异步的，相机收到抓拍指令不会立即返回图片等信息，而是相机抓拍后，会发送图片和香港信息给回调
 * 如果需要同步抓拍，请参考take_snapshot_sync函数
 * @param device_ip 相机IP
 * 返回API_STATUS::OK表示调用成功，否则返回FAILED或者一个正整数错误代码。
 */
int take_snapshot(const char * device_ip);

/**
 * @brief 抓拍图片，抓拍成功后，仅仅把抓拍的图片写入缓冲，回写图片的实际长度
 * 返回API_STATUS::OK表示调用成功，否则返回FAILED或者一个正整数错误代码。
 * @param device_ip 相机IP地址
 * @buff 图片待写入的buffer，写入的过程会自动分配内存, 调用端需要释放此内存
 * @len  写入图片的实际长度
 * @timeout 等待抓拍超时的限制, 默认30秒
 *
 */
int take_snapshot_sync(const char *device_ip, char **buff, int * len, int timeout = 30);

/**
 * @brief 抓拍图片到文件，API不对文件路径进行验证，也不支持非UTF8字符集的文件路径。
 * 	优先推荐使用take_snapshot_sync接口
 * 返回API_STATUS::OK表示调用成功，否则返回FAILED或者一个正整数错误代码。
 * @param device_ip 相机IP地址
 * @file 写入文件的名称
 * @param capacity buffer的容量
 * @param len  写入图片的实际长度
 * @param timeout 等待抓拍超时的限制，默认30秒
 *
 */
int take_snapshot_tofile(const char *device_ip, const char * file, int timeout=30);

/**
 * @brief 查询黑白名单, 查询成功把名单填入相关参数
 * @param device_ip 相机IP地址
 * @param type 名单类型 0 黑名单 1 白名单
 * @data  sBlackWhiteEntry数组
 * @cap   sBlackWhiteEntry数组的容量
 * @len   查询后的实际名单个数
 *
 * 注: 黑白名单最大数量不超过10000个.
 *
 */
int qurey_blackwhite_list(const char * device_ip, int type, sBlackWhiteEntry * data, int cap,  int *len);

/**
 * @brief 和qurey_blackwhite_list相同
 * 
 * @param device_ip 
 * @param type 
 * @param data 
 * @param cap 
 * @param len 
 * @return int 
 */
int qurey_blackwhite(const char * device_ip, int type, sBlackWhiteEntry * data, int cap,  int *len);

/**
 * @brief 查询黑白名单，查询成功后，为每个黑白名单调用BWList_CALLBACK
 * 
 * @param device_ip 相机IP地址
 * @param type 名单类型 0 黑名单 1 白名单
 * @param cb 回调函数
 * @return int 返回API_STATUS::OK表示调用成功，否则返回FAILED或者一个正整数错误代码。
 */
int query_blackwhite_list_callback(const char * device_ip, int type, BWList_CALLBACK cb);

/**
 * 添加黑白名单条目
 * @param device_ip 相机IP地址
 * @blackWhiteEntry 添加的黑白名单条目指针
 * @len				黑白名单条目的个数
 */
int append_blackwhite_list(const char * device_ip, const sBlackWhiteEntry * list, int len);

/**
 * @brief 删除黑白名单条目
 * @param device_ip 相机IP地址
 * @blackWhiteEntry 黑白名单条目指针
 * @len				黑白名单条目的个数
 */
int remove_blackwhite_list(const char * device_ip, const sBlackWhiteEntry * list, int len);

/**
 * @brief 清除相机中的黑白名单
 * @param type 0表示黑名单 1表示白名单
 *
 */
int clean_blackwhite_list(const char * device_ip, int type);

/**
 * get basic device information for device via IP, fill the DeviceInfo struct if return OK API_STATUS
 *
 * 获取设备的信息
 */
int get_device_info(const char * device_ip, DeviceInfo * pDeviceInfo);

/**
 * @brief setup certification key for a camera. once the certification key is set.
 * the camera will send confidential information encrypted by the certification key.
 * return OK if the camera has not setup a key.
 *
 * 给相机设置密钥，一旦给相机设置密钥后，相机立即对敏感的信息进行加密
 * 只有对没有设置过密钥的相机上调用此借口，才会返回OK。
 *
 *@param device_ip
 * @param key  the plain text of key, exclude space and invisible character. only support
 * 				16 characters of key.
 *              明文的密钥，不包括空格和非可见字符,
 * @param len  length of plain text of key, the value is 16
 *              密钥的长度，这里仅仅支持16个字符组成的密码。
 */
int setup_camera_cert_key(const char * device_ip, const char * key, int len);

#ifdef __cplusplus
#if __cplusplus
 }
#endif
#endif /* End of #ifdef __cplusplus */

#endif /* SRC_DEV_API_CLIENT_API_H_ */
