﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;

namespace Demo
{
    [StructLayout(LayoutKind.Sequential)]
    public struct PlateBox
    {
        public float xmin;   // 左上角x轴坐标
        public float ymin;   // 左上角y轴坐标
        public float xmax;   // 右下角x轴坐标
        public float ymax;   // 右下角y轴坐标
        public float score;  // 车牌框置信度
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct PlateBasicInfo
    {
        public int label;                                       // 识别的目标类型，小于等于1是车牌，大于1其他目标
        public float prob;                                      // 车牌识别置信度
        public PlateBox bbox;                                    // 车牌位置
        [MarshalAs(UnmanagedType.ByValArray, ArraySubType = UnmanagedType.I4, SizeConst = 12)]
        public float[] confidences;                              // 车牌号码单字置信度数组
        public int len;                                          // 车牌号码单字置信度数组长度
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 64)]
        public byte[] plateNos_;                                 // UTF8变长码编码的车牌号码。
        public int plateNoBytes;

        // 车牌号码实际实际长度，不超过MAX_PLATE_NO_BYTES变量定义
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
        public byte[] plateColor_;
        public int plateColorBytes;

        // 识别框编号，取值-1, 0, 1, 2,...，-1表示不支持此属性或者没有输出此属性
        public int anchorBoxNo;                 

                                                /**
                                             * 中国大陆车牌类型，定义如下
                                             * 0 {未知}
                                             * 10 - 16  {军 双层军牌 武警 双层武警牌 警 应急 新电动车}
                                             * 20 - 23 {黄 教练车牌 双层黄牌 挂车车牌}
                                             * 30 {蓝色小汽车}
                                             * 40 - 43 {使馆牌 领馆牌 粤港澳牌 黑色小汽车 }
                                             * 50 {大型双能源}
                                             * 60 - 63  {新能源 民航  农用车牌 电动车牌（非机动）}
                                             */
        public int plateType;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 15)]
        public byte[] deviceIp_;

        public int deviceIPbytes;

        /**
	 * 方向标志 -1 未定义 0 从上到下 1 从下到上
	 */
        public int direction;


        /**
	 * 车辆事件状态；对于不同系列有不同的定义
	 * 对低位视频桩、新能源车位管理相机、高位视频桩产品定义如下：
	 * 0    空
	 * 1	进场
	 * 2	停稳
	 * 3	出场
	 *
	 * 21	抓拍 (通用)
	 *
	 * E5系列定义的状态如下：
	 * 22 入场
	 * 23 倒车
	 * 24 机动车滞留
	 * 25 非机动车滞留
	 * 26 行人滞留
	 * 27 无牌车
	 * 28 跟车
	 *
	 */
       public int parkingSpotStatus;


        /**
	 * 车牌真伪标志 -1 未定义 0 假 1 真
	 * 
	 */
        public int plateFlag;

        /**
	 * 车头车尾标志 -1 未定义 0 车头 1 车尾
	 */
        public int headTailFlags;

        /**
	 *  E5系列产品有效
	 *  -1 未定义 0 相机视频触发 1 线圈触发 2 输入线路1 3 输入线路2 ...
	 *   
	 */
        public int triggerMode;


        public int imageBufferSize;       
        public IntPtr imageBuffer;

        public string plateNos
        {
            get
            {
                if (plateNoBytes > 0)
                {
                    return Encoding.UTF8.GetString(plateNos_, 0, plateNoBytes);
                }
                else {
                    return "";
                }
            }
        }

        public string plateColor
        {
            get
            {
                if (plateColorBytes > 0)
                {
                    return Encoding.UTF8.GetString(plateColor_, 0, plateColorBytes);
                }
                else
                {
                    return "";
                }
            }
        }

        public string deviceIP
        {
            get {
                if (deviceIPbytes > 0)
                {
                    return Encoding.UTF8.GetString(deviceIp_, 0, deviceIPbytes);
                }
                else
                {
                    return "";
                }
            }
        }
    };
    public enum API_STATUS
    {
        OK,
        FAILED
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct BlackWhiteEntry { 
        public int type ;  					// 0 黑名单 1 白名单
	    [MarshalAs(UnmanagedType.ByValArray, SizeConst = 64)]
        public byte[] plateNos_;                      // UTF8变长码编码的车牌号码。
        public int plateNoBytes;

        // 车牌号码实际实际长度，不超过MAX_PLATE_NO_BYTES变量定义
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
        public byte[] plateColor_;
        public int plateColorBytes;

	    public int startTime;						// 有效期开始的时间戳， -1表示永久有效
	    public int endTime;


        public string plateNos
        {
            get
            {
                if (plateNoBytes > 0)
                {
                    return Encoding.UTF8.GetString(plateNos_, 0, plateNoBytes);
                }
                else
                {
                    return "";
                }
            }
        }

        public string plateColor
        {
            get
            {
                if (plateColorBytes > 0)
                {
                    return Encoding.UTF8.GetString(plateColor_, 0, plateColorBytes);
                }
                else
                {
                    return "";
                }
            }
        }
    } 


    [StructLayout(LayoutKind.Sequential,CharSet=CharSet.Ansi,Pack=1)]
     struct DeviceInfo {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 64)]
		public byte[] deviceName; // 设备名称

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 32)]
		public byte[] ip;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
		public byte[] firmware; 	// 固件号

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
		public byte[] version;    // 版本
		[MarshalAs(UnmanagedType.ByValArray, SizeConst = 24)]
        public byte[] productNo ; // 产品序列号

		public int minVersion ;  // minor version
		public long startup ;     // 设备启动时间戳

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
		public byte[] productType;       // 产品型号
    } 
 
    static class LLPRAPI
    {

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void Callback(ref PlateBasicInfo pbi);

        // const char * ip, int code, const char * msg
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void CameraEventCallback(String ip, int code, String msg);

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void BlackWhiteListCallback(ref BlackWhiteEntry entry);

        // please change to the exact DLL file name
        const string DllName = "light-lpr-api-win64.dll";

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl,  EntryPoint = "init_client")]
        public extern static API_STATUS initClientApi(String localIP, int localPort, [MarshalAs(UnmanagedType.FunctionPtr)]Callback callback);

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "connect_device")]
        public extern static API_STATUS connectDevice(String cameraIP);

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl,  EntryPoint = "reg_camera_event_callback")]
        public extern static API_STATUS regCameraEventCallback(CameraEventCallback cameraEventCallback);

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "close_client")]
        public extern static API_STATUS closeClientApi();

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "set_time")]
        public extern static API_STATUS setTime(int enpoch);

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "trigger_alarmout")]
        public extern static API_STATUS triggerAlarmout(String deviceIP,  int line, int duration);

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "query_alarm")]
        public extern static API_STATUS queryAlarm(String deviceIP, int line, ref int status); 

       [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "write_uart")]
       public extern static API_STATUS writeUart(String deviceIP,  String alias, String data, int len);  //TOD
        
        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "take_snapshot")]
        public extern static API_STATUS takeSnapshot(String deviceIP); 

        //[DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "take_snapshot_sync")]
        //public extern static API_STATUS takeSnapshotSync(String deviceIP, [int, out]IntPtr duration, [in, out]int len, int timeout);

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "take_snapshot_tofile")]
        public extern static API_STATUS takeSnapshotToFile(String deviceIP, String file, int timeout); 

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "query_blackwhite_list_callback")]
        public extern static API_STATUS queryBlackwhiteListCallback(String deviceIP, int type, [MarshalAs(UnmanagedType.FunctionPtr)] BlackWhiteListCallback cb); 

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "append_blackwhite_list")]
        public extern static API_STATUS appendBlackwhiteList(String deviceIP, BlackWhiteEntry[] entries, int size); 

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "remove_blackwhite_list")]
        public extern static API_STATUS removeBlackwhiteList(String deviceIP, BlackWhiteEntry[] entries, int size);

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "clean_blackwhite_list")]
        public extern static API_STATUS cleanBlackwhiteList(String deviceIP, int type);   

       [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "get_device_info")]
       public extern static API_STATUS getDeviceInfo(String deviceIP, ref DeviceInfo deviceInfo);

       [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "set_cert_key")]
       public extern static API_STATUS setCertKey(String key, int len);

        [DllImport(DllName, CallingConvention = CallingConvention.Cdecl, EntryPoint = "setup_camera_cert_key")]
        public extern static API_STATUS setupCameraCertKey(String deviceIP, String key, int len); 
    } 
}
